﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Text.Json;
using System.IO;
using System.Linq;

namespace Ayakawa.DotNetRuntimeUpdateCheker
{
    class UpdateChecker
    {
        /// <summary>
        /// 本アプリが使用しているruntimeの種類とバージョンを保持する
        /// </summary>
        private static List<(string Name, Version Ver)> UsingRuntimeNames = new List<(string Name, Version Ver)>();

        /// <summary>
        /// UsingRuntimeNamesを初期化する
        /// </summary>
        public static void GetUsingRuntimeNames()
        {
            UsingRuntimeNames.Clear();

            // 現在のフレームワークのバージョンを取得を先に取得
            string currentFramework = RuntimeInformation.FrameworkDescription; // 例: ".NET 8.0.6"
            var currentParts = currentFramework.Split(' ');
            if (currentParts.Length < 2 || !Version.TryParse(currentParts[1], out var currentVersion))
            {
#if DEBUG
                Debug.WriteLine("AyaClock.WPF: 現在のフレームワークのバージョンが取得できませんでした。");
#endif
                return; // 現在のバージョンが取得できない場合は何もしない  
            }

            // runtimeconfig.jsonの存在を先に確認
            string exePath = Assembly.GetEntryAssembly().Location;
            string configPath = System.IO.Path.ChangeExtension(exePath, ".runtimeconfig.json");
            if (!File.Exists(configPath))
            {
#if DEBUG
                Debug.WriteLine("AyaClock.WPF:AyaClock.WPF: runtimeconfig.jsonが見つかりません。");
#endif
                return; // runtimeconfig.jsonが存在しない場合は何もしない
            }

            // runtimeconfig.jsonを読み込んで、使用しているフレームワークの名前とバージョンを取得
            // 但し、バージョンは初期値のままだと思われるので使用するのは先に取得したcurrentVersionを使用する   
            using var doc = JsonDocument.Parse(File.ReadAllText(configPath));
            var root = doc.RootElement;

            if (root.TryGetProperty("runtimeOptions", out var opts))
            {
                if (opts.TryGetProperty("frameworks", out var frameworks))
                {
                    foreach (var fw in frameworks.EnumerateArray())
                    {
                        if (Version.TryParse(fw.GetProperty("version").GetString(), out var _version))
                        {
                            UsingRuntimeNames.Add((fw.GetProperty("name").GetString(), currentVersion));
#if DEBUG
                            Debug.WriteLine("AyaClock.WPF: "+fw.GetProperty("name").GetString()+
                                              " " + _version.ToString() +
                                              " (current: " + currentVersion.ToString() + ")");   
#endif
                        }
                    }
                }
                else if (opts.TryGetProperty("framework", out var fw))
                {
                    if (Version.TryParse(fw.GetProperty("version").GetString(), out var _version))
                    {
                        UsingRuntimeNames.Add((fw.GetProperty("name").GetString(), currentVersion));
#if DEBUG
                        Debug.WriteLine("AyaClock.WPF: "+fw.GetProperty("name").GetString()+
                                          " " + _version.ToString() +
                                          " (current: " + currentVersion.ToString() + ")");   
#endif
                    }
                }
            }
        }

        /// <summary>
        /// 呼び出した時点でインストールされている.NET runtimeの一覧をdotnetコマンドを使用して取得
        /// dotnetコマンドが使えない場合は空のリストを返す。
        /// </summary>
        /// <returns>
        /// 一覧をList<string,Version>形式で返す
        /// </returns>
        private static List<(string Name, Version Ver)> GetInstalledDotnetRuntimes()
        {
            var result = new List<(string Name, Version Ver)>();

            ProcessStartInfo psi = new ProcessStartInfo
            {
                FileName = "dotnet",
                Arguments = "--list-runtimes",
                RedirectStandardOutput = true,
                UseShellExecute = false,
                CreateNoWindow = true
            };

            try
            {
                using var process = Process.Start(psi);
                using var reader = process.StandardOutput;
                string line;
                while ((line = reader.ReadLine()) != null)
                {
                    // 例: Microsoft.NETCore.App 6.0.29 [C:\...]
                    var parts = line.Split(' ');
                    if (parts.Length >= 2 && Version.TryParse(parts[1], out var version))
                    {
                        result.Add((parts[0], version));
#if DEBUG
                        Debug.WriteLine("AyaClock.WPF: installed runtime = " + parts[0] + " " + version.ToString());
#endif
                    }
                }
            }
            catch (Exception)
            {
                // do nothing
            }

            return result;
        }

        /// <summary>
        /// 値の意味 : None    変更無し or 分からない
        ///            Partial 一部updateされている
        ///            All     全てupdateされている
        /// </summary>
        public enum resultType
        {
            None = 0,
            Partial,
            All
        }

        /// <summary>
        /// 現在アプリケーションが使用している.NET runtimeのバージョンとインストールされている
        /// .NET runtimeのバージョンを比較し、現在使用中の全てのruntimeについてより新しいバージョン
        /// がインストールされていればresultType.Allを返す。一部がインストールされていれば
        /// resultType.Partialを返す。そうでなければresultType.Noneを返す。
        /// 何らかの理由で判定できなければresultType.Noneを返す。
        /// </summary>
        /// <returns>
        /// summary参照
        /// </returns>
        public static resultType IsExistsNewerVerRuntime()
        {
            // 使用中のruntimeが取得出来ていなければNoneを返す
            if (UsingRuntimeNames.Count == 0) return resultType.None;
            // インストール済みの.NET runtimeの一覧を取得
            var runtimes = GetInstalledDotnetRuntimes();
            if (runtimes.Count == 0) return resultType.None; // dotnetコマンドが使用できない場合はNoneを返す
                                                             // AyaClock.NETが使用している.NET runtimeのバージョンを取得
            string currentFramework = RuntimeInformation.FrameworkDescription; // 例: ".NET 8.0.0"
            var currentParts = currentFramework.Split(' ');
            if (currentParts.Length < 2 || !Version.TryParse(currentParts[1], out var currentVersion))
            {
                return resultType.None; // 現在のバージョンが取得できない場合はNoneを返す
            }
            // 同種の新しいランタイムがあるか判定
            int totalCount = 0;
            foreach (var (name, ver) in UsingRuntimeNames)
            {
                var newerRuntimes = runtimes
                    .Where(r => r.Name.Contains(name) &&
                                r.Ver.Major == ver.Major &&
                                r.Ver.Minor == ver.Minor &&
                                r.Ver.Build > ver.Build)
                    .OrderBy(r => r.Ver)
                    .ToList();
                totalCount += newerRuntimes.Count;
            }
#if DEBUG
            Debug.WriteLine("AyaClock.WPF: using runtimes=" + UsingRuntimeNames.Count +
                              ", totalCount=" + totalCount);
#endif
            // 一つも更新されていない場合
            if (totalCount == 0)
                return resultType.None;
            // 部分的に更新されている場合
            if (totalCount < UsingRuntimeNames.Count)
                return resultType.Partial;
            // 全部更新されている場合
            return resultType.All;
        }
    }
}