﻿using System;
using System.Text;
using System.Windows;
using System.Windows.Interop;
using System.Runtime.InteropServices;
using Microsoft.Win32.SafeHandles;

//
// 参照 https://qiita.com/YSRKEN/items/b8072f4816a88924605d
//
//

namespace Ayakawa.MonitorTools
{

    // MonitorFromWindowが返したディスプレイの種類
    public enum MonitorDefaultTo { Null, Primary, Nearest }
    // GetDpiForMonitorが返したDPIの種類
    enum MonitorDpiType { Effective, Angular, Raw, Default = Effective }

    [StructLayout(LayoutKind.Sequential, Pack = 4)]
    public struct POINT
    {
        public Int32 x;
        public Int32 y;
    }

    [StructLayout(LayoutKind.Sequential, Pack = 4)]
    public struct RECT
    {
        public Int32 left;
        public Int32 top;
        public Int32 right;
        public Int32 bottom;
    }

    static class MONITORINFO_Consts
    {
        public const int MONITORINFOF_PRIMARY = 1;
    }

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Auto, Pack = 4)]
    public struct MONITORINFO
    {
        public UInt32 cbSize;
        public RECT rcMonitor;
        public RECT rcWork;
        public UInt32 dwFlags;
    }

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Auto, Pack = 4)]
    unsafe public struct MONITORINFOEX
    {
        public UInt32 cbSize;
        public RECT rcMonitor;
        public RECT rcWork;
        public UInt32 dwFlags;
        public fixed char szDevice[32];
    }

    class NativeMethods
    {
        [DllImport("user32.dll", CharSet = CharSet.Unicode, SetLastError = true)]
        public static extern IntPtr MonitorFromWindow(IntPtr hwnd, MonitorDefaultTo dwFlags);
        [DllImport("user32.dll", CharSet = CharSet.Unicode, SetLastError = true)]
        public static extern IntPtr MonitorFromPoint(POINT pt, MonitorDefaultTo dwFlags);
        // ディスプレイハンドルからDPIを取得
        [DllImport("SHCore.dll", CharSet = CharSet.Unicode, PreserveSig = false)]
        public static extern void GetDpiForMonitor(IntPtr hmonitor, MonitorDpiType dpiType, ref uint dpiX, ref uint dpiY);
        //

        [DllImport("user32.dll", SetLastError = true)]
        [return: MarshalAs(UnmanagedType.Bool)]
        public static extern bool GetMonitorInfo(IntPtr hmonitor, ref MONITORINFO mi);

        [DllImport("user32.dll", CharSet = CharSet.Unicode, SetLastError = true)]
        [return: MarshalAs(UnmanagedType.Bool)]
        public static extern bool GetMonitorInfo(IntPtr hmonitor, ref MONITORINFOEX mi);

        [DllImport("user32.dll", CharSet = CharSet.Unicode)]
        public extern static IntPtr FindWindow(string lpClassName, string lpWindowName);

        unsafe public delegate bool EnumWindowsDelegate(IntPtr hWnd, IntPtr lparam);

        [DllImport("user32.dll")]
        [return: MarshalAs(UnmanagedType.Bool)]
        public extern static bool EnumWindows([MarshalAs(UnmanagedType.FunctionPtr)] EnumWindowsDelegate lpEnumFunc, [MarshalAs(UnmanagedType.SysUInt), In, Out] IntPtr lparam);

        [DllImport("user32.dll", CharSet = CharSet.Unicode, SetLastError = true)]
        public static extern int GetClassName(IntPtr hWnd, [MarshalAs(UnmanagedType.LPWStr), Out] StringBuilder lpClassName, int nMaxCount);

        [DllImport("User32.Dll", SetLastError = true)]
        [return: MarshalAs(UnmanagedType.Bool)]
        public static extern bool GetWindowRect(IntPtr hWnd, ref RECT rect);

    }

    // ここまで

    class HelperTools
    {
        public static bool GetMonitorInfoFromPoint(POINT pt, MonitorDefaultTo dwFlags, ref MONITORINFO mi)
        {
            // 座標を使ってモニターを検出
            var hmonitor = NativeMethods.MonitorFromPoint(pt, dwFlags);
            if (hmonitor == IntPtr.Zero)
                // モニタを検出できなかった
                return false;
            // モニター情報を取得
            mi.cbSize = (UInt32)Marshal.SizeOf(typeof(MONITORINFO));
            return NativeMethods.GetMonitorInfo(hmonitor, ref mi);
        }

        public static bool GetMonitorInfoFromWindow(Window AWindow, MonitorDefaultTo dwFlags, ref MONITORINFO mi)
        {
            // window handleを使ってモニターを検出
            var hwnd = new WindowInteropHelper(AWindow).Handle;
            if (hwnd==IntPtr.Zero)
            {
                // AWindowが無効だった場合は左上の座標を基準に取得
                POINT pt;
                pt.x = (int)AWindow.Left; pt.y = (int)AWindow.Top;
                return GetMonitorInfoFromPoint(pt, dwFlags, ref mi);
            }
            var hmonitor = NativeMethods.MonitorFromWindow(hwnd, dwFlags);
            if (hmonitor == IntPtr.Zero)
                return false;
            // モニター情報を取得
            mi.cbSize = (UInt32)Marshal.SizeOf(typeof(MONITORINFO));
            return NativeMethods.GetMonitorInfo(hmonitor, ref mi);
        }

        public static bool GetMonitorInfoFromPoint(POINT pt, MonitorDefaultTo dwFlags, ref MONITORINFOEX mi)
        {
            // 座標を使ってモニターを検出
            var hmonitor = NativeMethods.MonitorFromPoint(pt, dwFlags);
            if (hmonitor == IntPtr.Zero)
                return false;
            // モニター情報を取得
            mi.cbSize = (UInt32)Marshal.SizeOf(typeof(MONITORINFOEX));
            return NativeMethods.GetMonitorInfo(hmonitor, ref mi);
        }

        public static bool GetMonitorInfoFromWindow(Window AWindow, MonitorDefaultTo dwFlags, ref MONITORINFOEX mi)
        {
            // window handleを使ってモニターを検出
            var hwnd = new WindowInteropHelper(AWindow).Handle;
            if (hwnd==IntPtr.Zero)
            {
                POINT pt;
                pt.x = (int)AWindow.Left; pt.y = (int)AWindow.Top;
                return GetMonitorInfoFromPoint(pt, dwFlags, ref mi);
            }
            var hmonitor = NativeMethods.MonitorFromWindow(hwnd, dwFlags);
            if (hmonitor == IntPtr.Zero)
                return false;
            // モニター情報を取得
            mi.cbSize = (UInt32)Marshal.SizeOf(typeof(MONITORINFOEX));
            return NativeMethods.GetMonitorInfo(hmonitor, ref mi);
        }

        public static void GetDpiFromPoint(POINT pt, MonitorDefaultTo dwFlags, MonitorDpiType dpiType, ref uint dpiX, ref uint dpiY)
        {
            // 座標を使ってモニターを検出
            var hmonitor = NativeMethods.MonitorFromPoint(pt, dwFlags);
            if (hmonitor == IntPtr.Zero)
                return;
            // DPIを取得
            NativeMethods.GetDpiForMonitor(hmonitor, dpiType, ref dpiX, ref dpiY);
        }
        public static void GetDpiFromWindow(Window AWindow, MonitorDefaultTo dwFlags, MonitorDpiType dpiType, ref uint dpiX, ref uint dpiY)
        {
            // window handleを使ってモニターを検出
            var hwnd = new WindowInteropHelper(AWindow).Handle;
            if (hwnd==IntPtr.Zero)
            {
                POINT pt;
                pt.x = (int)AWindow.Left; pt.y = (int)AWindow.Top;
                GetDpiFromPoint(pt, dwFlags, dpiType, ref dpiX, ref dpiY);
                return;
            }    
            var hmonitor = NativeMethods.MonitorFromWindow(hwnd, dwFlags);
            // DPIを取得
            if (hmonitor!=IntPtr.Zero)
                NativeMethods.GetDpiForMonitor(hmonitor, dpiType, ref dpiX, ref dpiY);
        }

        // GetTaskBarRectFromWindowのヘルパー
        static private bool GetPrimaryTaskbarRect(ref RECT rect)
        {
            bool result = false;
            IntPtr hwnd = NativeMethods.FindWindow("Shell_TrayWnd", null);
            if (hwnd != IntPtr.Zero)
            {
                result = NativeMethods.GetWindowRect(hwnd, ref rect);
            }
            return result;
        }

        [StructLayout(LayoutKind.Sequential, Pack = 4)]
        private struct TRAYRECTS
        {
            public RECT SrcRect, TrayRect;
            public bool Found;
        }

        // EnumWindows用callback関数
        unsafe static private bool EWFunc(IntPtr hWnd, IntPtr lparam)
        {
            bool result = true;

            StringBuilder cn = new StringBuilder(256);
            if (0 != NativeMethods.GetClassName(hWnd, cn, cn.Capacity))
            {
                // クラス名が取得出来たので、Taskbarかどうかをチェック
                if (cn.ToString().Equals("Shell_SecondaryTrayWnd"))
                {
                    // Taskbarだったので領域取得
                    IntPtr ip = (IntPtr)lparam;
                    TRAYRECTS tr = (TRAYRECTS)Marshal.PtrToStructure(ip, typeof(TRAYRECTS));
                    if (NativeMethods.GetWindowRect(hWnd, ref tr.TrayRect))
                    {
                        // 領域取得成功したので、目的のモニタ上のtaskbarかどうかを判定
                        if (tr.TrayRect.right < tr.SrcRect.left ||
                            tr.TrayRect.left > tr.SrcRect.right ||
                            tr.TrayRect.bottom < tr.SrcRect.top ||
                            tr.TrayRect.top > tr.SrcRect.bottom)
                        {
                            // モニタとtaskbarの領域が全く重なっていなかった
                            tr.Found = false;
                        }
                        else
                        {
                            // 範囲内だったので目的のtaskber!!
                            tr.Found = true;
                            Marshal.StructureToPtr(tr, ip, false); // ここで書き戻す！
                            result = false;  // ここで探索打ち切り
                        }
                    }
                }
            }

            return result;
        }

        // ここまで

        public static bool GetTaskBarRectFromWindow(Window AWindow, ref RECT ATaskBarRect)
        {
            // モニター情報取得
            MONITORINFO mi = new MONITORINFO();
            if (!GetMonitorInfoFromWindow(AWindow, MonitorDefaultTo.Nearest, ref mi))
                return false; // モニター情報取得失敗!!
            // 
            if ((mi.dwFlags & MONITORINFO_Consts.MONITORINFOF_PRIMARY) !=0)
            {
                // primary monitorだった
                return GetPrimaryTaskbarRect(ref ATaskBarRect);
            }
            else
            {
                // primary monitor以外だった
                TRAYRECTS trs = new TRAYRECTS();
                trs.SrcRect = mi.rcMonitor;
                trs.Found = false;
                unsafe
                {
                    try
                    {
                        IntPtr ip = Marshal.AllocHGlobal(Marshal.SizeOf(typeof(TRAYRECTS)));
                        try
                        {
                            Marshal.StructureToPtr(trs, ip, false);
                            NativeMethods.EnumWindowsDelegate cb = new NativeMethods.EnumWindowsDelegate(EWFunc);
                            NativeMethods.EnumWindows(cb, ip);
                            trs = (TRAYRECTS)Marshal.PtrToStructure(ip, typeof(TRAYRECTS)); // ここで書き戻す
                        }
                        finally
                        {
                            Marshal.FreeHGlobal(ip);
                        }
                    }
                    catch (OutOfMemoryException)
                    {
                        // メモリ不足の場合、見つからなかった扱いにしておく
                    }
                }
                if (trs.Found)
                {
                    ATaskBarRect = trs.TrayRect;
                    return true;
                }
                else
                    return false;
            }
        }
    }
}